<?php

namespace App\Http\Controllers;

use App\BusinessLocation;
use App\Contact;
use App\CustomerGroup;
use App\Reservation;
use App\User;
use App\Rooms;
use App\Utils\Util;
use DB;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Yajra\DataTables\Facades\DataTables;
use App\Utils\AccomodationUtil;

class ReservationController extends Controller
{
    /**
     * All Utils instance.
     *
     */
    protected $commonUtil;
    protected $restUtil;

    public function __construct(Util $commonUtil, AccomodationUtil $restUtil)
    {
        $this->commonUtil = $commonUtil;
        $this->restUtil = $restUtil;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if (!auth()->user()->can('crud_all_reservations') && !auth()->user()->can('crud_own_reservations')) {
            abort(403, 'Unauthorized action.');
        }
        $business_id = request()->session()->get('user.business_id');
        $rooms_id = request()->session()->get('user.rooms_id');
        $user_id = request()->has('user_id') ? request()->user_id : null;
        if (!auth()->user()->hasPermissionTo('crud_all_reservations') && !$this->restUtil->is_admin(auth()->user(), $business_id)) {
            $user_id = request()->session()->get('user.id');
        }
        if (request()->ajax()) {
            $filters = [
                'start_date' => request()->start,
                'end_date' => request()->end,
                'user_id' => $user_id,
                'rooms_id' => $rooms_id,
                'location_id' => !empty(request()->location_id) ? request()->location_id : null,
                'business_id' => $business_id
            ];

            $events = $this->restUtil->getReservationsForCalendar($filters);

            return $events;
        }

        $business_locations = BusinessLocation::forDropdown($business_id);

        $customers =  Contact::customersDropdown($business_id, false);

        $correspondents = User::forDropdown($business_id, false);

        $rooms = Rooms::forDropdown($business_id, false);

        $types = Contact::getContactTypes();
        $customer_groups = CustomerGroup::forDropdown($business_id);

        return view('reservation.index', compact('business_locations', 'customers', 'correspondents','rooms', 'types', 'customer_groups'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if (!auth()->user()->can('crud_all_reservations') && !auth()->user()->can('crud_own_reservations')) {
            abort(403, 'Unauthorized action.');
        }
        try {
            if ($request->ajax()) {
                $business_id = request()->session()->get('user.business_id');
                $user_id = request()->session()->get('user.id');
                $rooms_id = request()->session()->get('user.rooms.id');

                $input = $request->input();
                $reservation_start = $this->commonUtil->uf_date($input['reservation_start'], true);
                $reservation_end = $this->commonUtil->uf_date($input['reservation_end'], true);
                $date_range = [$reservation_start, $reservation_end];

                //Check if reservation is available for the required input
                $query = Reservation::where('business_id', $business_id)
                                ->where('location_id', $input['location_id'])
                                ->where('contact_id', $input['contact_id'])
                                ->where(function ($q) use ($date_range) {
                                    $q->whereBetween('reservation_start', $date_range)
                                    ->orWhereBetween('reservation_end', $date_range);
                                });

                if (isset($input['res_table_id'])) {
                    $query->where('table_id', $input['res_table_id']);
                }

                if (isset($input['rooms_id'])) {
                    $query->where('rooms_id', $input['rooms_id']);
                }
                
                $existing_reservation = $query->first();
                if (empty($existing_reservation)) {
                    $input['business_id'] = $business_id;
                    $input['created_by'] = $user_id;
                    $input['rooms_id'] = $rooms_id;
                    $input['reservation_start'] = $reservation_start;
                    $input['reservation_end'] = $reservation_end;
                    $reservation = Reservation::createReservation($input);
                    
                    $output = ['success' => 1,
                        'msg' => trans("lang_v1.added_success"),
                    ];

                    //Send notification to customer
                    if (isset($input['send_notification']) && $input['send_notification'] == 1) {
                        $output['send_notification'] = 1;
                        $output['notification_url'] = action('NotificationController@getTemplate', ["transaction_id" => $reservation->id,"template_for" => "new_reservation"]);
                    }
                } else {
                    $time_range = $this->commonUtil->format_date($existing_reservation->reservation_start, true) . ' ~ ' .
                                    $this->commonUtil->format_date($existing_reservation->reservation_end, true);

                    $output = ['success' => 0,
                            'msg' => trans(
                                "accomodation.reservation_not_available",
                                ['customer_name' => $existing_reservation->customer->name,
                                'reservation_time_range' => $time_range]
                            )
                        ];
                }
            } else {
                die(__("messages.something_went_wrong"));
            }
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile(). "Line:" . $e->getLine(). "Message:" . $e->getMessage());
            $output = ['success' => 0,
                            'msg' => __("messages.something_went_wrong")
                        ];
        }
        return $output;
    }

    /**
     * Display the specified resource.
     *
     * @param  \int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $reservation = Reservation::where('business_id', $business_id)
                                ->where('id', $id)
                                ->with(['table', 'rooms','customer', 'correspondent', 'waiter', 'location'])
                                ->first();
            if (!empty($reservation)) {
                $reservation_start = $this->commonUtil->format_date($reservation->reservation_start, true);
                $reservation_end = $this->commonUtil->format_date($reservation->reservation_end, true);

                $reservation_statuses = [
                    'waiting' => __('lang_v1.waiting'),
                    'booked' => __('room.booked'),
                    'completed' => __('room.completed'),
                    'cancelled' => __('room.cancelled'),
                ];
                return view('reservation.show', compact('reservation', 'reservation_start', 'reservation_end', 'reservation_statuses'));
            }
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Reservation  $reservation
     * @return \Illuminate\Http\Response
     */
    public function edit(Reservation $reservation)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Reservation  $reservation
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        if (!auth()->user()->can('crud_all_reservations') && !auth()->user()->can('crud_own_reservations')) {
            abort(403, 'Unauthorized action.');
        }
        try {
            $business_id = $request->session()->get('user.business_id');
            $reservation = Reservation::where('business_id', $business_id)
                                ->find($id);
            if (!empty($reservation)) {
                $reservation->reservation_status = $request->reservation_status;
                $reservation->save();
            }

            $output = ['success' => 1,
                            'msg' => trans("lang_v1.updated_success")
                        ];
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile(). "Line:" . $e->getLine(). "Message:" . $e->getMessage());
            $output = ['success' => 0,
                            'msg' => __("messages.something_went_wrong")
                        ];
        }
        return $output;
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Reservation  $reservation
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if (!auth()->user()->can('crud_all_reservations') && !auth()->user()->can('crud_own_reservations')) {
            abort(403, 'Unauthorized action.');
        }
        try {
            $business_id = request()->session()->get('user.business_id');
            $reservation = Reservation::where('business_id', $business_id)
                                ->where('id', $id)
                                ->delete();
            $output = ['success' => 1,
                            'msg' => trans("lang_v1.deleted_success")
                        ];
        } catch (\Exception $e) {
            \Log::emergency("File:" . $e->getFile(). "Line:" . $e->getLine(). "Message:" . $e->getMessage());
            $output = ['success' => 0,
                            'msg' => __("messages.something_went_wrong")
                        ];
        }
        return $output;
    }

    /**
     * Retrieves todays reservations
     *
     * @param  \App\Reservation  $reservation
     * @return \Illuminate\Http\Response
     */
    public function getTodaysReservations()
    {
        if (!auth()->user()->can('crud_all_reservations') && !auth()->user()->can('crud_own_reservations')) {
            abort(403, 'Unauthorized action.');
        }

        if (request()->ajax()) {
            $business_id = request()->session()->get('user.business_id');
            $user_id = request()->session()->get('user.id');
            $query = Reservation::where('business_id', $business_id)
                        ->where('reservation_status', 'booked')
                        ->with(['table','rooms', 'customer', 'correspondent', 'waiter', 'location']);
                        
            if (!empty(request()->location_id)) {
                $query->where('location_id', request()->location_id);
            }

            if (!auth()->user()->hasPermissionTo('crud_all_reservations') && !$this->commonUtil->is_admin(auth()->user(), $business_id)) {

                $query->where(function ($query) use ($user_id){
                    $query->where('created_by', $user_id)
                        ->orWhere('correspondent_id', $user_id)
                        ->orWhere('waiter_id', $user_id);
                });

                //$query->where('created_by', $user_id);
            }

            return Datatables::of($query)
            
                ->editColumn('table', function ($row) {
                    return !empty($row->table->name) ? $row->table->name : '--';
                })
                ->editColumn('rooms', function ($row) {
                    return !empty($row->rooms->name) ? $row->rooms->name : '--';
                })
                ->editColumn('customer', function ($row) {
                    return !empty($row->customer->name) ? $row->customer->name : '--';
                })
                ->editColumn('correspondent', function ($row) {
                    return !empty($row->correspondent->user_full_name) ? $row->correspondent->user_full_name : '--';
                })
                ->editColumn('waiter', function ($row) {
                    return !empty($row->waiter->user_full_name) ? $row->waiter->user_full_name : '--';
                })
                ->editColumn('location', function ($row) {
                    return !empty($row->location->name) ? $row->location->name : '--';
                })
                ->editColumn('reservation_start', function ($row) {
                    return $this->commonUtil->format_date($row->reservation_start, true);
                })
                ->editColumn('reservation_end', function ($row) {
                    return $this->commonUtil->format_date($row->reservation_end, true);
                })
               ->removeColumn('id')
                ->make(true);
        }
    }
}



